/*
 * Written by Dawid Kurzyniec and released to the public domain, as explained
 * at http://creativecommons.org/licenses/publicdomain
 */

package edu.emory.mathcs.util.io;

import java.io.*;
import edu.emory.mathcs.backport.java.util.concurrent.*;

/**
 * Input source that supports read with timeouts. If timeout occurs before
 * any data is available, TimeoutException is thrown.
 *
 * @author Dawid Kurzyniec
 * @version 1.0
 */
public interface TimedInput extends Input {

    /**
     * Read a byte of data if available within specified timeout.
     *
     * @param timeout the number of milliseconds to wait for data
     * @return the byte of data read
     * @throws IOException if an I/O error occurs
     * @throws TimeoutException if timeout occurs before data is available
     *
     * @see InputStream#read()
     */
    int timedRead(long timeout) throws IOException, TimeoutException;

    /**
     * Reads some number of bytes of data from the input stream into
     * an array of bytes.  An attempt is made to read as many as
     * <code>buf.length</code> bytes, but a smaller number may be read.
     * The number of bytes actually read is returned as an integer.
     * If <code>buf</code> has nonzero size but no data is available within the
     * specified timeout and the stream is not at EOF, TimeoutException is
     * thrown.
     *
     * @param buf the buffer into which the data is read
     * @param timeout the maximum number of milliseconds to wait before data
     *                is available
     * @return the actual number of bytes read
     * @throws IOException if an I/O error occurs
     * @throws TimeoutException if timeout occurs before data is available
     *
     * @see InputStream#read(byte[])
     */
    int timedRead(byte[] buf, long timeout) throws IOException, TimeoutException;

    /**
     * Reads up to <code>len</code> bytes of data from the input stream into
     * an array of bytes.  An attempt is made to read as many as
     * <code>len</code> bytes, but a smaller number may be read.
     * The number of bytes actually read is returned as an integer.
     * If nonzero bytes is requested but no data is available within the
     * specified timeout and the stream is not at EOF, TimeoutException is
     * thrown.
     *
     * @param buf the buffer into which the data is read
     * @param off the start offset in array <code>buf</code> where data
     *            is written
     * @param len the maximum number of bytes to read
     * @param timeout the maximum number of milliseconds to wait before data
     *                is available
     * @return the actual number of bytes read
     * @throws IOException if an I/O error occurs
     * @throws TimeoutException if timeout occurs before data is available
     *
     * @see InputStream#read(byte[],int,int)
     */
    int timedRead(byte[] buf, int off, int len, long timeout) throws IOException,
                                                                     TimeoutException;
}
